﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

using System.IO;
using DocumentFormat.OpenXml.Packaging;
using DocumentFormat.OpenXml.Spreadsheet;
using DocumentFormat.OpenXml;

using Framework;

namespace Framework.Tools.OpenXmlHelpers
{
    public class XlsxBook : IDisposable
    {
        private SpreadsheetDocument _doc = null;
        private WorkbookPart _wbPart = null;
        private WorkbookView _wbView = null;
        private SharedStringTablePart _sstPart = null;

        private MemoryStream _stream = null;

        public bool IsClosed = false;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public XlsxBook()
        {
            this.Create(null);
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public XlsxBook(string path)
        {
            using (var stream = File.OpenRead(path))
            {
                this.Create(stream);
            }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public XlsxBook(Stream stream)
        {
            this.Create(stream);
        }

        /// <summary>
        /// 空のブックを作成します
        /// </summary>
        private void Create(Stream stream)
        {
            _stream = new System.IO.MemoryStream();

            if (stream == null)
            {
                //空のブックを作成する
                _doc = SpreadsheetDocument.Create(_stream, SpreadsheetDocumentType.Workbook, false);
                this.InitBook();

                //新規シートを追加する
                _activeSheet = this.AddSheet();
            }
            else
            {
                //ストリームをコピーする
                CopyStream(stream, _stream);

                //既存のブックを開く
                _doc = SpreadsheetDocument.Open(_stream, true);
                this.InitBook();
            }
        }

        /// <summary>
        /// ブックの初期設定
        /// </summary>
        private void InitBook()
        {
            //ワークブックを追加する
            if (_doc.WorkbookPart == null)
            {
                _doc.AddWorkbookPart();
                _doc.WorkbookPart.Workbook = new Workbook();
            }
            _wbPart = _doc.WorkbookPart;

            //WorkbookViewを追加する
            if (_wbPart.Workbook.BookViews == null)
            {
                _wbPart.Workbook.AppendChild<BookViews>(new BookViews());
            }
            if (_wbPart.Workbook.BookViews.Count() == 0)
            {
                _wbPart.Workbook.BookViews.AppendChild<WorkbookView>(new WorkbookView());
            }
            _wbView = _wbPart.Workbook.BookViews.GetFirstChild<WorkbookView>();

            //SharedStringTableを追加する
            if (_wbPart.SharedStringTablePart == null)
            {
                _wbPart.AddNewPart<SharedStringTablePart>();
                _wbPart.SharedStringTablePart.SharedStringTable = new SharedStringTable();
            }
            _sstPart = _wbPart.SharedStringTablePart;

            //名前定義を追加する
            if (_wbPart.Workbook.DefinedNames == null)
            {
                _wbPart.Workbook.DefinedNames = new DefinedNames();
            }

            //Sheetsを追加する
            if (_wbPart.Workbook.Sheets == null)
            {
                _wbPart.Workbook.Sheets = new Sheets();
            }
        }

        /// <summary>
        /// 保存して終了します
        /// </summary>
        private void Close()
        {
            if (this.IsClosed == true)
            {
                return;
            }

            Sheets().Foreach(t => t.Save());
            _sstPart.SharedStringTable.Save();
            _wbPart.Workbook.Save();
            _doc.Close();

            this.IsClosed = true;
        }

        /// <summary>
        /// ファイルに保存して終了します
        /// </summary>
        public void SaveAs(string path)
        {
            this.Close();
 
            _stream.Position = 0;
            using (var fileStream = File.Create(path))
            {
                CopyStream(_stream, fileStream);
            }
        }

        /// <summary>
        /// 終了して、結果を配列で返します
        /// </summary>
        public byte[] ToArray()
        {
            this.Close();

            return _stream.ToArray();
        }

        /// <summary>
        /// アクティブシート
        /// </summary>
        public XlsxSheet ActiveSheet
        {
            get
            {
                if (_activeSheet == null)
                {

                    _activeSheet = Sheets().Skip((int)_wbView.ActiveTab.Value).FirstOrDefault();
                }
                return _activeSheet;
            }
            set
            {
                var sheet = Sheets().WithIdx().FirstOrDefault(t=>t._Data.SheetName == value.SheetName);
                if (sheet != null)
                {
                    _wbView.ActiveTab = (uint)sheet._Index;
                    _activeSheet = value;
                }
            }
        }
        private XlsxSheet _activeSheet = null;

        /// <summary>
        /// 指定されたシートを返します。
        /// </summary>
        public XlsxSheet Sheets(int index)
        {
            return Sheets().Skip<XlsxSheet>(index).FirstOrDefault();
        }

        /// <summary>
        /// 指定されたシートを返します。
        /// </summary>
        public XlsxSheet Sheets(string sheetName)
        {
            return Sheets().Where(t => t.SheetName == sheetName).FirstOrDefault();
        }

        /// <summary>
        /// シートを列挙します。
        /// </summary>
        /// <returns></returns>
        public IEnumerable<XlsxSheet> Sheets()
        {
            foreach (var sheet in _wbPart.Workbook.Sheets.Elements<Sheet>())
            {
                yield return new XlsxSheet(this, GetWorksheetPart(sheet), sheet);
            }
        }

        /// <summary>
        /// シートを追加します。
        /// </summary>
        public XlsxSheet AddSheet()
        {
            return AddSheet(null);
        }

        /// <summary>
        /// シートを追加します。
        /// </summary>
        public XlsxSheet AddSheet(string sheetName)
        {
            var wsPart = _wbPart.AddNewPart<WorksheetPart>();
            wsPart.Worksheet = new Worksheet(new SheetData());


            uint sheetId = 1;
            var sheets = _wbPart.Workbook.Sheets;
            if (sheets.Elements<Sheet>().Any() == true)
            {
                sheetId = sheets.Elements<Sheet>().Select(t => t.SheetId.Value).Max() + 1;
            }
            
            var sheet = new Sheet()
            {
                Id = _wbPart.GetIdOfPart(wsPart),
                SheetId = sheetId,
                Name = "Sheet{0}".Fmt(sheetId),
            };
            sheets.Append(sheet);
            
            //シート名変更
            var xlsxSheet = new XlsxSheet(this, wsPart, sheet);
            if (sheetName != null)
            {
                xlsxSheet.SheetName = sheetName;
            }

            //シートをアクティブ化
            this.ActiveSheet = xlsxSheet;

            return xlsxSheet;
        }

        /// <summary>
        /// 指定された共有文字列を返します。
        /// </summary>
        public string SharedString(int index)
        {
            var text = _sstPart.SharedStringTable.ElementAt(index).InnerText;
            return text;
        }

        /// <summary>
        /// 文字列を共有文字列テーブルに追加し、インデックスを返します。
        /// </summary>
        public int AddSharedString(string text)
        {
            int idx = 0;
            foreach (SharedStringItem item in _sstPart.SharedStringTable.Elements<SharedStringItem>())
            {
                if (item.InnerText == text)
                {
                    return idx;
                }
                idx++;
            }

            _sstPart.SharedStringTable.AppendChild(new SharedStringItem(new Text(text)));

            return idx;
        }

        /// <summary>
        /// 指定された名前定義を返しいます。
        /// </summary>
        public DefinedName DefinedNames(string name)
        {
            return _wbPart.Workbook.DefinedNames
                .Elements<DefinedName>()
                .FirstOrDefault(t => t.Name.Value == name);
        }

        /// <summary>
        /// アクティブシートの指定されたセルを返します。
        /// </summary>
        public XlsxCell Cell(string cellReference)
        {
            return this.ActiveSheet.Cell(cellReference);
        }

        /// <summary>
        /// アクティブシートの指定されたセルを返します。
        /// </summary>
        public XlsxCell Cell(int columnIndex, int rowIndex)
        {
            return this.ActiveSheet.Cell(columnIndex, rowIndex);
        }

        /// <summary>
        /// （内部使用）WorksheetPartを返します。
        /// </summary>
        private WorksheetPart GetWorksheetPart(Sheet sheet)
        {
            return (WorksheetPart)_wbPart.GetPartById(sheet.Id);
        }

        internal static void CopyStream(Stream input, Stream output)
        {
            var buf = new byte[8 * 1024];
            int len;
            while ((len = input.Read(buf, 0, buf.Length)) > 0)
            {
                output.Write(buf, 0, len);
            }
            output.Flush();
        }

        /// <summary>
        /// Dispose
        /// </summary>
        void IDisposable.Dispose()
        {
            if (_doc != null)
            {
                _doc.Dispose();
            }

            if (_stream != null)
            {
                _stream.Close();
                _stream.Dispose();
            }
        }
    }
}
